<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class UserController extends Controller
{
    public function signup(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'name'      => 'required|string|max:255',
                'email'     => 'required|string|email|max:255|unique:users',
                'password'  => 'required|string|min:8'
            ]);

            $user = User::create([
                'name'  => $request->name,
                'email' => $request->email,
                'password'  => bcrypt($request->password)
            ]);

            return response()->json([
                'status'    => 'Ok',
                'message'   => 'Successfully created user'
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'status'    => 'Error',
                'message'   => 'Validation failed',
                'errors'    => $e->errors()
            ], 422);
        }
    }

    public function login(Request $request)
    {
        // Validar las credenciales del usuario
        $request->validate([
            'email'     => 'required|string|email',
            'password'  => 'required|string'
        ]);

        // Intentar autenticar al usuario
        $credentials = $request->only('email', 'password');
        
        if (Auth::attempt($credentials)) {
            // Obtener el usuario autenticado con la relación country
            $user = User::with(['country', 'locality'])->find(Auth::id());

            // Generar un token para el usuario usando Passport
            $token = $user->createToken('Token Name')->accessToken;

            if ($user->active == 0) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Usuario inactivo',
                ], 401);
            }

            // Devolver una respuesta con el token
            return response()->json([
                'status' => 'success',
                'message' => 'Login successful',
                'user' => $user,
                'access_token' => $token,
                'token_type' => 'Bearer',
            ]);
        } else {
            // Devolver una respuesta de error si las credenciales son incorrectas
            return response()->json([
                'status' => 'error',
                'message' => 'Acceso no autorizado',
            ], 401);
        }
    }

    public function getAllAdminUsersCountry()
    {
        $users = User::where('id_rol', 2)->with('country')->get();
        return response()->json([
            'status' => 'success',
            'users' => $users
        ],200);
    }

    public function getAllAdminUsersLocality($country_id)
    {
            $users = User::where('id_rol', 3) // Rol administrador de sede
                ->whereHas('locality.country', function ($query) use ($country_id) {
                    $query->where('id', $country_id);
                })
                ->with('locality')
                ->get();


        return response()->json([
            'status' => 'success',
            'users' => $users
        ], 200);
    }

    public function getAllUsersByLocality($locality_id)
    {
            $users = User::where('id_rol', 4) // Rol gastos
            ->where('locality_id', $locality_id)->get();


        return response()->json([
            'status' => 'success',
            'users' => $users
        ], 200);
    }

    public function createAdminUserCountry(Request $request)
    {
        $validatedData = $request->validate([
            'name'      => 'required|string|max:255',
            'last_name' => 'required',
            'email'     => 'required|string|email|max:255|unique:users',
            'country_id'=> 'required|exists:countries,id'
        ]);

        $user = User::create([
            'name'      => $validatedData['name'],
            'last_name' => $validatedData['last_name'],
            'email'     => $validatedData['email'],
            'password'  => bcrypt('gastos123'),
            'id_rol'    => 2,
            'country_id'=> $validatedData['country_id'],
            'active'    => $request->active
        ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'Usuario creado con éxito',
            'user'    => $user
        ], 200);
    }

    public function createAdminUserLocality(Request $request)
    {
        $validatedData = $request->validate([
            'name'      => 'required|string|max:255',
            'last_name' => 'required',
            'email'     => 'required|string|email|max:255|unique:users',
            'locality_id'=> 'required|exists:localities,id'
        ]);

        $user = User::create([
            'name'      => $validatedData['name'],
            'last_name' => $validatedData['last_name'],
            'email'     => $validatedData['email'],
            'password'  => bcrypt('gastos123'),
            'id_rol'    => 3, // rol administrador de sede
            'locality_id'=> $validatedData['locality_id'],
            'active'    => $request->active
        ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'Usuario creado con éxito',
            'user'    => $user
        ], 200);
    }

    public function createUser(Request $request)
    {
        $validatedData = $request->validate([
            'name'      => 'required|string|max:255',
            'last_name' => 'required',
            'email'     => 'required|string|email|max:255|unique:users',
            'locality_id'=> 'required|exists:localities,id'
        ]);

        $user = User::create([
            'name'      => $validatedData['name'],
            'last_name' => $validatedData['last_name'],
            'email'     => $validatedData['email'],
            'password'  => bcrypt('gastos123'),
            'id_rol'    => 4, // rol Gastos
            'locality_id'=> $validatedData['locality_id'],
            'active'    => $request->active
        ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'Usuario creado con éxito',
            'user'    => $user
        ], 200);
    }

    public function updateAdminUserCountry(Request $request, $id)
    {
        $validatedData = $request->validate([
            'name'      => 'sometimes|required|string|max:255',
            'last_name' => 'sometimes|required|string|max:255',
            'email'     => 'sometimes|required|string|email|max:255|unique:users,email,' . $id,
            'country_id'=> 'sometimes|required|exists:countries,id',
        ]);

        $user = User::where('id_rol', 2)->findOrFail($id);

            $user->name = $request->name;
            $user->last_name = $request->last_name;
            $user->email = $request->email;
            $user->country_id = $request->country_id;
            $user->active = $request->active;
        $user->save();

        return response()->json([
            'status'  => 'success',
            'message' => 'Usuario actualizado con éxito',
            'user'    => $user
        ], 200);
    }

    public function updateAdminUserLocality(Request $request, $id)
    {
        $validatedData = $request->validate([
            'name'      => 'sometimes|required|string|max:255',
            'last_name' => 'sometimes|required|string|max:255',
            'email'     => 'sometimes|required|string|email|max:255|unique:users,email,' . $id,
            'locality_id'=> 'sometimes|required|exists:localities,id',
        ]);

        $user = User::where('id_rol', 3)->findOrFail($id);

            $user->name = $request->name;
            $user->last_name = $request->last_name;
            $user->email = $request->email;
            $user->locality_id = $request->locality_id;
            $user->active = $request->active;
        $user->save();

        return response()->json([
            'status'  => 'success',
            'message' => 'Usuario actualizado con éxito',
            'user'    => $user
        ], 200);
    }

    public function updateUser(Request $request, $id)
    {
        $validatedData = $request->validate([
            'name'      => 'sometimes|required|string|max:255',
            'last_name' => 'sometimes|required|string|max:255',
            'email'     => 'sometimes|required|string|email|max:255|unique:users,email,' . $id,
            'locality_id'=> 'sometimes|required|exists:localities,id',
        ]);

        $user = User::where('id_rol', 4)->findOrFail($id);

            $user->name = $request->name;
            $user->last_name = $request->last_name;
            $user->email = $request->email;
            $user->locality_id = $request->locality_id;
            $user->active = $request->active;
        $user->save();

        return response()->json([
            'status'  => 'success',
            'message' => 'Usuario actualizado con éxito',
            'user'    => $user
        ], 200);
    }
    
}
